#include "functor_tbl.h"

namespace lp {

	data::data(const char* ptr) : tag(strlen(ptr)+1)
	{
		// Dynamic
		s = static_cast<char*>(malloc(tag));
		if (!s) throw std::bad_alloc();
		memcpy(s,ptr,tag);
		assert(s != ptr && strcmp(ptr,s) == 0);
	}

	data::data(const string& str) : tag(str.length()+1) 
	{
		s = static_cast<char*>(malloc(tag));
		if (!s) throw std::bad_alloc();
		memcpy(s,str.c_str(),tag);
		s[tag-1] = '\0';
		assert(str == s);
	}

	data::data(const data& dat) : tag(dat.tag) 
	{
		switch (tag) {
		case 0: break;
		case data::int_tag: i = dat.i; break;
		case data::float_tag: d = dat.d; break;
		default:
			s = static_cast<char*>(malloc(tag));
			if (!s) throw std::bad_alloc();
			memcpy(s,dat.s,tag);
			assert(s != dat.s && strcmp(s,dat.s) == 0);
		}
	}

	data& data::operator=(const data& dat) 
	{
		if (this != &dat) {
			if (tag > 0) {
				// Optimization opportunities
				if (dat.tag > 0 && tag >= dat.tag) {
					if (tag != dat.tag) {
						// realloc down
#pragma warning(suppress: 6308)
						s = static_cast<char*>(realloc(s,dat.tag));
						if (!s) throw std::bad_alloc();
						tag = dat.tag;
					} // else: no (re)allocation needed, just copy
					memcpy(s,dat.s,dat.tag);
					assert(s != dat.s && strcmp(s,dat.s) == 0);
					return *this;
				} else {
					free(s); // free dynamic memory, continue
				}
			}

			tag = dat.tag;
			switch (dat.tag) {
			case 0: break;
			case data::int_tag: i = dat.i; break;
			case data::float_tag: d = dat.d; break;
			default:
				s = static_cast<char*>(malloc(dat.tag));
				if (!s) throw std::bad_alloc();
				memcpy(s,dat.s,dat.tag);
				assert(s != dat.s && strcmp(s,dat.s) == 0);
			}
		}
		return *this;
	}

	data::data(data&& dat) : tag(dat.tag)
	{
		switch (dat.tag) {
		case 0: break;
		case data::int_tag: i = dat.i; break;
		case data::float_tag: d = dat.d; break;
		default:
			s = dat.s; 
			dat.s = nullptr;
		}
	}

	data& data::operator=(data&& dat) 
	{
		if (this != &dat) {
			if (tag > 0) free(s);
			tag = dat.tag;
			switch (dat.tag) {
			case 0: break;
			case data::int_tag: i = dat.i; break;
			case data::float_tag: d = dat.d; break;
			default: 
				s = dat.s; 
				dat.s = nullptr;
			}
		}
		return *this;
	}


	int data::compare(const data& dat) const
	{
		if (tag != dat.tag) return -2;
		switch (tag) {
		case 0: return true;
		case data::int_tag: return i < dat.i ? -1 : (i > dat.i ? 1 : 0);
		case data::float_tag: return d < dat.d ? -1 : (d > dat.d ? 1 : 0);
		default: 
			{
				const auto r = strcmp(s,dat.s);
				return r < 0 ? -1 : (r > 0 ? 1 : 0);
			}
		}
	}


	bool data::operator<(const data& dat) const
	{
		const auto r = this->compare(dat);
		if (r == -2) return false;
		return r < 0;
	}

	bool data::operator>(const data& dat) const
	{
		const auto r = this->compare(dat);
		if (r == -2) return false;
		return r > 0;
	}

	bool data::operator<=(const data& dat) const
	{
		const auto r = this->compare(dat);
		if (r == -2) return false;
		return r <= 0;
	}

	bool data::operator>=(const data& dat) const
	{
		const auto r = this->compare(dat);
		if (r == -2) return false;
		return r >= 0;
	}


	ostream& operator<<(ostream& os, const data& d) 
	{
		switch (d.type()) {
		case 0: os << "nil"; return os;
		case data::int_tag: os << d.get_int(); return os;
		case data::float_tag: os << d.get_float(); return os;
		default: os << d.get_atom(); return os;
		}
	}

	int numeric_compare(const data& d1, const data& d2)
	{
		switch (d1.type()) {
		case data::int_tag:
			switch (d2.type()) {
			case data::int_tag: return d1.get_int() < d2.get_int() ? -1 : d1.get_int() > d2.get_int() ? 1 : 0;
			case data::float_tag: return d1.get_int() < d2.get_float() ? -1 : d1.get_int() > d2.get_float() ? 1 : 0;
			}
		case data::float_tag:
			switch (d2.type()) {
			case data::int_tag: return d1.get_float() < d2.get_int() ? -1 : d1.get_float() > d2.get_int() ? 1 : 0;
			case data::float_tag: return d1.get_float() < d2.get_float() ? -1 : d1.get_float() > d2.get_float() ? 1 : 0;
			}
		}
		throw domain_error("incommensurable objects in functor numeric comparison\n");
	}



	functor_map::functor_map() 
		: vc(INT_MAX/2), fc(INT_MIN/2), rvc(0), rfc(0)
	{
		add(pair_id,".");
		add(empty_list_id,"[]");
		add(sequence_id,",");
		add(if_id,":-");
		add(plus_id,"+");
		add(minus_id,"-");
		add(mul_id,"*");
		add(div_id,"/");
		add(pow_id,"^");
		add(hash_id,"#");
		add(cut_id,"!");
		add(naf_id,"\\+");
		add(not_id,"not");
		add(or_id,";");
		add(query_id,"?-");
		add(equal_id,"=");
		add(unequal_id,"\\=");
		add(lessat_id,"@<");
		add(lesseqat_id,"@=<");
		add(greaterat_id,"@>");
		add(greatereqat_id,"@>=");
		add(compare_id,"compare");
		add(isvariant_id,"=@=");
		add(isnotvariant_id,"\\=@=");
		add(syn_equal_id,"==");
		add(syn_unequal_id,"\\==");
		add(is_id,"is");
		add(math_equal_id,"=:=");
		add(math_unequal_id,"=\\=");
		add(lt_id,"<");
		add(gt_id,">");
		add(le_id,"=<");
		add(ge_id,">=");
		add(univ_id,"=..");
		add(lp_id,"(");
		add(rp_id,")");
		add(ll_id,"[");
		add(rl_id,"]");
		add(lsep_id,"|");
		add(zero_id,0LL);
		add(s_id,"s");
		add(any_id,"any");
		add(true_id,"true");
		add(fail_id,"fail");
		add(false_id,"false");
		add(var_id,"var");
		add(nonvar_id,"nonvar");
		add(atomic_id,"atomic");
		add(number_id,"number");
		add(functor_id,"functor");
		add(arg_id,"arg");
		add(set_id,"set");
		add(halt_id,"halt");
		add(op_id,"op");
		add(listing_id,"listing");
		add(listing_wam_id,"listing_wam");
		add(display_id,"display");
		add(consult_id,"consult");
		add(assert_id,"assert");
		add(asserta_id,"asserta");
		add(nb_linkarg_id,"nb_linkarg");
		add(duplicate_term_id,"duplicate_term");
		add(assertz_id,"assertz");
		add(retract_id,"retract");
		add(retractall_id,"retractall");
		//add(compress_id,"compress");
		add(bottom_id,"bottom");
		add(generalize_id,"generalize");
		add(modeh_id,"modeh");
		add(modeb_id,"modeb");
		add(or2_id,"_;");
		add(qmark_id,"?");
		add(generalise_id,"generalise");
		add(one_id,1LL);
		add(unset_id,"unset");
		add(get_id,"get");
		add(constant_id,"constant");
		add(atom_id,"atom");
		add(findall_id,"findall");
		add(bagof_id,"bagof");
		add(bagof2_id,"_bagof");
		add(sort_id,"sort");
		add(setof_id,"setof");
		add(clause_id,"clause");
		add(once_id,"once");
		add(integer_id,"integer");
		add(int_id,"int");
		add(float_id,"float");
		add(ifthen_id,"->");
		add(star2_id,"**");
		add(ifbranch_id,"_->");
		add(write_id,"write");
		add(nl_id,"nl");
		add(nrsample_id,"nrsample");
		add(id_nrsample_id,"id_nrsample");
		add(emulate_nrsample_id,"emulate_nrsample");
		add(qg_id,"qg");
		add(qgga_id,"qgga");
		add(functional_id,"functional");
		add(reflexive_id,"reflexive");
		add(irreflexive_id,"irreflexive");
		add(symmetric_id,"symmetric");
		add(antisymmetric_id,"antisymmetric");
		add(transitive_id,"transitive");
		add(idempotent_id,"idempotent");
		add(implies_id,"=>");
		add(prevents_id,"prevents");
		add(prevent_id,"prevent");
		add(equal2_id,"_=");
		add(sample_id,"sample");
		add(label_id,"label");
		add(nat_id,"nat");
		add(vsids_id,"vsids");
		add(svsids_id,"svsids");
		add(random_id,"random");
		add(determination_id,"determination");
		add(include_predicate_id,"include_predicate");
		add(exclude_predicate_id,"exclude_predicate");
		add(evalfn_id,"evalfn");
		add(compression_id,"compression");
		add(posonly_id,"posonly");
		add(minset_id,"minset");
		add(heuristic_id,"heuristic");
		add(enumerate_id,"enumerate");
		add(name_id,"name");
		add(call_id,"call");
	}


	bool functor_map::is_variable(const string& s)
	{
		if (s.empty()) return false; // s == ""
		else if (isupper(s.front())) return true; // s == "X..."
		else if (s.front() != '_') return false; // s == "f..."
		else if (s.length() == 1) return true; // s == "_"
		else return isupper(s[1]) || s[1] == '_'; // s == "_X..." || s == "__..."
	}

	data functor_map::get_data(id_type i) const
	{
		auto at = dmap.find(i);
		if (at != dmap.end()) {
			// Found
			return data(*at->second);
		} else {
			// Not found, this is a computer generated id (e.g. _G12)
			if (i > 0) {
				return data("_G" + stringify(i));
			} else if (i < 0) {
				return data("_f" + stringify(i));
			} else {
				return data("_nil");
			}
		}
	}

	const char* functor_map::get_atom(id_type i) const
	{
		auto at = dmap.find(i);
		if (at != dmap.end() && at->second->is_atom()) {
			// Found
			return at->second->get_atom();
		} else {
			throw conversion_error();
		}
	}

	long long functor_map::get_int(id_type i) const
	{
		auto at = dmap.find(i);
		if (at != dmap.end() && at->second->is_int()) {
			// Found
			return at->second->get_int();
		} else {
			throw non_numeric();
		}
	}

	double functor_map::get_float(id_type i) const
	{
		auto at = dmap.find(i);
		if (at != dmap.end() && at->second->is_double()) {
			// Found
			return at->second->get_float();
		} else {
			throw non_numeric();
		}
	}

	data functor_map::get_numeric(id_type i) const
	{
		auto at = dmap.find(i);
		if (at != dmap.end() && at->second->is_numeric()) {
			return data(*at->second); // Found
		} else {
			throw non_numeric();
		}
	}

	id_type functor_map::id(const string& s)
	{
		// it is deemed too dangerous to store "_" since they are not real vars
		assert( s != "_" );

		data tmp(s);
		auto at = imap.find(tmp);
		if (at != imap.end()) {
			return at->second;
		} else {
			// symbol not added yet, add it
			assert(vc < numeric_limits<decltype(vc)>::max());
			assert(fc > numeric_limits<decltype(fc)>::min());
			const id_type idx = is_variable(s) ? ++vc : --fc;

			auto j = imap.insert(make_pair(std::move(tmp),idx)).first;
			//cerr << "inserted: " << &j->first << " -> " << j->first << "\n";
			dmap.insert(make_pair(idx,&j->first));
			return idx;
		}
	}

	id_type functor_map::id(const char* s)
	{
		// it is deemed too dangerous to store "_" since they are not real vars
		assert( strcmp(s,"_") != 0 );

		data tmp(s);
		auto at = imap.find(tmp);
		if (at != imap.end()) {
			//cerr << "looked up string: " << s << " <-> " << at->first << " <-> " << get_data(at->second) << "\n";
			return at->second;
		} else {
			// symbol not added yet, add it
			assert(vc < numeric_limits<decltype(vc)>::max());
			assert(fc > numeric_limits<decltype(fc)>::min());
			const id_type idx = is_variable(s) ? ++vc : --fc;

			auto j = imap.insert(make_pair(std::move(tmp),idx)).first;
			//cerr << "inserted: " << &j->first << " -> " << j->first << "\n";
			dmap.insert(make_pair(idx,&j->first));
			return idx;
		}
	}


	id_type functor_map::id(long long val)
	{
		auto at = imap.find(val);
		if (at != imap.end()) {
			return at->second;
		} else {
			// symbol not added yet, add it
			assert(fc > numeric_limits<decltype(fc)>::min());
			--fc; // integers are non-variables
			auto j = imap.insert(make_pair(data(val),fc)).first;
			//cerr << "inserted: " << &j->first << " -> " << j->first << "\n";
			dmap.insert(make_pair(fc,&j->first));
			return fc;
		}
	}

	id_type functor_map::id(double val)
	{
		auto at = imap.find(val);
		if (at != imap.end()) {
			return at->second;
		} else {
			// symbol not added yet, add it
			assert(fc > numeric_limits<decltype(fc)>::min());
			--fc; // doubles are non-variables
			auto j = imap.insert(make_pair(data(val),fc)).first;
			//cerr << "inserted: " << &j->first << " -> " << j->first << "\n";
			dmap.insert(make_pair(fc,&j->first));
			return fc;
		}
	}

	id_type functor_map::id(const data& d)
	{
		switch (d.type()) {
		case 0: return 0;
		case data::int_tag: return this->id(d.get_int());
		case data::float_tag: return this->id(d.get_float());
		default: return this->id(d.get_atom());
		}
	}

	id_type functor_map::id(data&& d)
	{
		switch (d.type()) {
		case 0: return 0;
		case data::int_tag: return this->id(d.get_int());
		case data::float_tag: return this->id(d.get_float());
		default: return this->id(std::move(d.get_atom()));
		}
	}

	void functor_map::add(id_type i, const char* s) 
	{
		auto p = imap.insert(make_pair(data(s),i));
		if (p.second) {
			assert(p.second == true);
			//cerr << "adding: " << &p.first->first << " -> " << p.first->first << "\n";
			dmap.insert(make_pair(i,&p.first->first));
		}
	}

	void functor_map::add(id_type i, long long v) 
	{
		auto p = imap.insert(make_pair(data(v),i));
		if (p.second) {
			assert(p.second == true);
			//cerr << "adding: " << &p.first->first << " -> " << p.first->first << "\n";
			dmap.insert(make_pair(i,&p.first->first));
		}
	}

	void functor_map::add(id_type i, double v) 
	{
		auto p = imap.insert(make_pair(v,i));
		if (p.second) {
			assert(p.second == true);
			//cerr << "adding: " << &p.first->first << " -> " << p.first->first << "\n";
			dmap.insert(make_pair(i,&p.first->first));
		}
	}

	void functor_map::print(ostream& os, id_type i) const
	{
		auto at = dmap.find(i);
		if (at != dmap.end()) {
			os << *at->second; // found it, print
		} else {
			if (i > 0) {
				os << "_G" << i;
			} else if (i < 0) {
				os << "_f" << -i;
			} else {
				os << "_nil";
			}
		}
	}


}


